/*----------------------------------------------------------------------------*/
/* Copyright (c) FIRST 2008. All Rights Reserved.							  */
/* Open Source Software - may be modified and shared by FRC teams. The code   */
/* must be accompanied by the FIRST BSD license file in $(WIND_BASE)/WPILib.  */
/*                                                                            */
/* Modified by Alex Morozov for the robot simulator program                   */
/* Note that only one gyro is supported at this time.  No matter how many     */
/* gyros you instantiate, they all will return the same reading.              */
/*----------------------------------------------------------------------------*/

#include "Gyro.h"
#include "RobotMath.h"
#include "SimInterface.h"

/**
 * Initialize the gyro.
 * Calibrate the gyro by running for a number of samples and computing the center value for this
 * part. Then use the center value as the Accumulator center value for subsequent measurements.
 * It's important to make sure that the robot is not moving while the centering calculations are
 * in progress, this is typically done when the robot is first turned on while it's sitting at
 * rest before the competition starts.
 */
void Gyro::InitGyro()
{
    Reset();
}

/**
 * Gyro constructor given a slot and a channel.
 * 
 * @param slot The cRIO slot for the analog module the gyro is connected to.
 * @param channel The analog channel the gyro is connected to.
 */
Gyro::Gyro(UINT32 slot, UINT32 channel)
{
    m_channel = channel;
	InitGyro();
}

/**
 * Gyro constructor with only a channel.
 * 
 * Use the default analog module slot.
 * 
 * @param channel The analog channel the gyro is connected to.
 */
Gyro::Gyro(UINT32 channel)
{
    m_channel = channel;
	InitGyro();
}

/**
 * Gyro constructor with a precreated analog channel object.
 * Use this constructor when the analog channel needs to be shared. There
 * is no reference counting when an AnalogChannel is passed to the gyro.
 * @param channel The AnalogChannel object that the gyro is connected to.
 */
//Gyro::Gyro(AnalogChannel *channel)
//{
//    InitGyro();
//}

//Gyro::Gyro(AnalogChannel &channel)
//{
//	InitGyro();
//}

/**
 * Reset the gyro.
 * Resets the gyro to a heading of zero. This can be used if there is significant
 * drift in the gyro and it needs to be recalibrated after it has been running.
 */
void Gyro::Reset()
{
    m_reset_value = SimInterface::GetAnalog( m_channel );
}

/**
 * Delete (free) the accumulator and the analog components used for the gyro.
 */
Gyro::~Gyro()
{
}

/**
 * Return the actual angle in degrees that the robot is currently facing.
 * 
 * The angle is based on the current accumulator value corrected by the oversampling rate, the
 * gyro type and the A/D calibration values.
 * The angle is continuous, that is can go beyond 360 degrees. This make algorithms that wouldn't
 * want to see a discontinuity in the gyro output as it sweeps past 0 on the second time around.
 * 
 * @return the current heading of the robot in degrees. This heading is based on integration
 * of the returned rate from the gyro.
 */
float Gyro::GetAngle( void )
{
    float current_value = SimInterface::GetAnalog( m_channel );
    return( ( current_value - m_reset_value ) * 180.0f / float( PI ) );
}


/**
 * Set the gyro type based on the sensitivity.
 * This takes the number of volts/degree/second sensitivity of the gyro and uses it in subsequent
 * calculations to allow the code to work with multiple gyros.
 * 
 * @param voltsPerDegreePerSecond The type of gyro specified as the voltage that represents one degree/second.
 */
void Gyro::SetSensitivity( float voltsPerDegreePerSecond )
{
}

