/*
 *  SimInterface.cpp
 *  CppBot
 *
 *  Created by Alex on 1/31/10.
 *  Copyright 2010 __MyCompanyName__. All rights reserved.
 *
 */

#include "SimInterface.h"

#include <stdio.h>

#include "RobotBase.h"
#include "RobotParameter.h"

float SimInterface::m_analog[ ANALOG_CNT ];
bool SimInterface::m_dio[ DIO_CNT ];
double SimInterface::m_encoders[ PWM_CNT ];
SimInterface::joystick_data_type SimInterface::m_joysticks[ JOYSTICK_CNT ];
float SimInterface::m_pwm[ PWM_CNT ];
SimInterface::sim_object_vector_type SimInterface::m_objects;


/**************************************************************************
* GetJoystickAxis - returns the current deflection along the specified axis
**************************************************************************/
float SimInterface::GetJoystickAxis
    ( 
    int                 joystick_idx,
    joystick_axis_type  axis_idx 
    )
{
if( joystick_idx < JOYSTICK_CNT )
    {
    switch( axis_idx )
        {
        case X_AXIS:
            return( m_joysticks[ joystick_idx ].x );
            break;
            
        case Y_AXIS:
            return( m_joysticks[ joystick_idx ].y );
            break;
        
        case Z_AXIS:
            return( m_joysticks[ joystick_idx ].z );
            break;

        default:
            printf( "SimInterface::GetJoystickAxis: axis %d is not valid", axis_idx );
            return( 0.0f );
            break;
        }
    }
else
    {
    printf( "SimInterface::GetJoystickAxis: joystick %d does not exist", joystick_idx );
    return( 0.0f );
    }
}


/**************************************************************************
* GetJoystickButton - returns true if the specified button is pressed, or 
*   false otherwise.
**************************************************************************/
bool SimInterface::GetJoystickButton( int joystick_idx, unsigned int button_idx )
{
if( joystick_idx < JOYSTICK_CNT )
    {
    if( m_joysticks[ joystick_idx ].button_mask & ( 1 << button_idx ) )
        {
        return( true );
        }
    else
        {
        return( false );
        }
    }
else
    {
    printf( "joystick %d does not exist", joystick_idx );
    return( false );
    }
}   /* SimInterface::GetJoystickButton */


/**************************************************************************
**************************************************************************/
float SimInterface::GetPWM( unsigned int channel )
{
if( channel < PWM_CNT )
    {
    return( m_pwm[ channel ] );
    }
else
    {
    printf( "SimInterface::GetPWM: invalid channel %d\n", channel );
    return( 0.0f );
    }
}   /* SimInterface::GetPWM */


/**************************************************************************
**************************************************************************/
void SimInterface::Register( SimObject * p_object )
{
m_objects.push_back( p_object );
}

/**************************************************************************
* SetJoystick - called by the GLUTInterface to update the state of the 
*   joystick.
**************************************************************************/
void SimInterface::SetJoystick
    ( 
    unsigned int        n,          /* joystick number: 0, 1, 2, ...        */
    float               x,          /* x axis value [-1,+1]                 */
    float               y,          /* y axis value [-1,+1]                 */
    float               z,          /* z axis value [-1,+1]                 */
    unsigned int button_mask        /* bitmask of pressed buttons           */
    )
{
if( n < JOYSTICK_CNT )
    {
    m_joysticks[ n ].x = x;
    m_joysticks[ n ].y = y;
    m_joysticks[ n ].z = z;
    m_joysticks[ n ].button_mask = button_mask;
    }
}   /* SimInterface::SetJoystick */


/**************************************************************************
* SetPWM - called by the WPI library wrappers to set PWM values.
**************************************************************************/
void SimInterface::SetPWM( unsigned int channel, float value )
{
if( channel < PWM_CNT )
    {
    if( value > 1.0f )
        {
        printf( "SimInterface::SetPWM: truncating %f to 1.0f\n", value );
        value = 1.0f;
        }
    else if( value < -1.0f )
        {
        printf( "SimInterface::SetPWM: truncating %f to -1.0f\n", value );
        value = -1.0f;
        }
        
    m_pwm[ channel ] = value;
    }
else
    {
    printf( "SimInterface::SetPWM: invalid channel %d\n", channel );
    }
}   /* SimInterface::SetPWM */


/**************************************************************************
* Update - periodic update function
**************************************************************************/
void SimInterface::Update( unsigned int update_rate_ms )
{
    // Update line following sensor states
    // This code assumes that the line is the ray starting at origin or pointing up
    for( int i = 0; i < RobotBase::LFS_CNT; i++ )
    {
        const PositionVector& lfs_pos_fc( RobotBase::p_robot->GetLfsPosFc( i ) );
        bool sensor_value = false;
        if( lfs_pos_fc.y >= 0 )
        {
            double distance_to_line = fabs( lfs_pos_fc.x );
            if( distance_to_line < 2.5 )
            {
                sensor_value = true;
            }
            else if( distance_to_line < 5 )
            {
                sensor_value = ( distance_to_line < ( rand() % 5 ) );                              
            }
        }   
        
        SetDIO( LEFT_LF_CHANNEL + i, sensor_value ); 
    }
    
    // Update encoders.
	// !!! THIS HAS BEEN DEPRECATED !!!
	// !!! Use IncrementEncoder() instead !!!
    // TODO: Currently, encoder channel must match the PWM channel.  Fix this.
    //for( int i = 0; i < PWM_CNT; i++ )
    //{
    //    double delta = m_pwm[ i ] * 100;
    //    m_encoders[ i ] += delta;
	//}

    // Update simulated objects
    for( sim_object_vector_type::iterator i = m_objects.begin( ); i < m_objects.end( ); i++ )
    {
        (*i)->Update( update_rate_ms );
    }
        
}
