/*********************************************************************
*
*   RobotBase.cpp - Robot base class
*
*   Copyright:
*       Software source code by Alex Morozov and Chris D. Locke is
*       licensed under a Creative Commons Attribution-Noncommercial-
*       Share Alike 3.0 United States License
*       (http://creativecommons.org/licenses/by-nc-sa/3.0/us/)
*
*********************************************************************/

#include <stdio.h>
#include <math.h>

#include "RobotBase.h"
#include "RobotMath.h"
#include "RobotParameter.h"
#include "SimInterface.h"

/*
 * Class static variables
 */
RobotBase * RobotBase::p_robot = NULL;


/**************************************************************************
* SetMode()
*   Set operational mode (disabled, autonomous, periodic)
**************************************************************************/
void RobotBase::SetMode( rob_mode_type mode )
{
    m_mode = mode;
    switch( mode )
    {
    case ROB_MODE_DISABLED:
        DisabledInit( );
        break;
        
    case ROB_MODE_AUTO:
        AutonomousInit( );
        break;
        
    case ROB_MODE_TELEOP:
        TeleopInit( );
        break;
        
    default:
        _ASSERT( 0 );
        break;
    }
}


void RobotBase::Update
    ( 
    unsigned int period_ms          // update rate in milliseconds
    )
{
    m_update_rate_ms = period_ms;

    // Get motor values
    // TODO: remove motor variables and use GetPWM directly
    m_leftMotor = SimInterface::GetPWM( 1 );
    m_rightMotor = SimInterface::GetPWM( 2 );

    // Call the derived class periodic function
    switch( m_mode )
    {
    case ROB_MODE_DISABLED:
        DisabledPeriodic( );
        break;
    
    case ROB_MODE_AUTO:
        AutonomousPeriodic( );
        break;
        
    case ROB_MODE_TELEOP:
        TeleopPeriodic( );
        break;
        
    default:
        _ASSERT( 0 );
        break;
    }

    MoveRobot( period_ms );
}


RobotBase::RobotBase( ) : 
    m_mode( ROB_MODE_DISABLED )
{
    m_leftMotor = 0.0;
    m_rightMotor = 0.0;
    m_x = 0.0;
    m_y = 0.0;
    m_yaw = 0.0;
    m_width = 50;                   // pretend its wider so that it doesn't turn so fast
    leftMotorAccumulator = 0.0;
    rightMotorAccumulator = 0.0;

    //m_logFile = fopen( "robotLog.csv", "w" );
    //fprintf( m_logFile, "time, x, y, yaw, left motor, right motor\n" );

    //s_debugFile = fopen( "debugLog.csv", "w" );
    //fprintf( s_debugFile, "rotation angle, rotation origin,,rotation arm, rotation arm check\n" );

    RobotBase::p_robot = this;
    
    // Set line following sensor positions in the body coordinate system
    m_lfs_pos_bc[ LFS_LEFT ].x = -5.0;
    m_lfs_pos_bc[ LFS_CENTER ].x = 0.0;
    m_lfs_pos_bc[ LFS_RIGHT ].x = 5.0;
    for( int i = 0; i < LFS_CNT; i++ )
    {
        m_lfs_pos_bc[ i ].y = 10.0;
    }
    UpdateLFSPos();
}


RobotBase::~RobotBase( )
{
}


void RobotBase::MoveRobot( unsigned int period_ms )
{
    // Scale motor speed so that at full speed the robot moves one grid every two seconds
    // One grid is 100 units
    // Note that the right motor is usually connected backwards, such that 
    // positive speed will cause the robot to go back
    double scale = 50.0 * period_ms / 1000.0;
    double leftMotor  = m_leftMotor * scale;
    double rightMotor = -m_rightMotor * scale;
    leftMotorAccumulator += leftMotor;
    rightMotorAccumulator += rightMotor;

    // If both wheels are turning at the same speed, then move the robot
    // straight
    if( fabs( leftMotor - rightMotor ) < EPSILON )
        {
        double theta = ConvertYawToTheta( m_yaw );
        double dx = leftMotor * cos( theta );
        double dy = leftMotor * sin( theta );
        m_x = m_x + dx;
        m_y = m_y + dy;
        }

    // If wheels are turning at different speeds, then the robot will turn
    else
        {
        double rotationAngle;
        double rotationDistance;
        ComputeRotationAngle( leftMotor,
                              rightMotor,
                              m_width,
                              &rotationAngle,
                              &rotationDistance );

        // Determine rotation axis coordinates by rotating
        // (rotationDistance,0) vector around the robot position through
        // the yaw angle
        double xo = m_x + rotationDistance;
        double yo = m_y;
        RotateAboutPoint( &xo, &yo, m_x, m_y, m_yaw );

        //fprintf( s_debugFile,
        //         "%f, %f, %f, %f, %f\n",
        //       rotationAngle * 180.0 / PI,
        //       xo,
        //       yo,
        //       rotationDistance,
        //       sqrt( ( m_x - xo ) * ( m_x - xo ) + ( m_y - yo ) * ( m_y - yo ) ) );
        RotateAboutPoint( &m_x, &m_y, xo, yo, rotationAngle );
        m_yaw = fmod( m_yaw + rotationAngle, 2 * PI );
        
        // Update Gyro
        float gyro = SimInterface::GetAnalog( YAW_GYRO_CHANNEL );
        gyro = gyro + float( rotationAngle );
        SimInterface::SetAnalog( YAW_GYRO_CHANNEL, gyro );
        }

    UpdateLFSPos(); 
}


void RobotBase::SetMotor( double * p_motor, double speed )
{
    if( speed < -1.0 )
    {
        *p_motor = -1.0;
    }
    else if( speed > 1.0 )
    {
        *p_motor = 1.0;
    }
    else
    {
        *p_motor = speed;
    }
}


/**************************************************************************
* GetWatchdog()
*   TODO
**************************************************************************/
Watchdog& RobotBase::GetWatchdog()
{
    return( m_watchdog );
}


/******************************************************************************
* Cycle through different motor settings
******************************************************************************/
void RobotBase::TestPeriodic( unsigned int t_ms )
{
if( ( t_ms % 1000 ) == 0 )
    {
    m_rightMotor += 0.5;
    if( m_rightMotor > 1.1 )
        {
        m_rightMotor = -1.0;
        m_leftMotor += 0.5;
        if( m_leftMotor > 1.1 )
            {
            m_leftMotor = -1.0;
            }
        }
    }
}
