/*
 *  GLUTInterface.cpp
 *  CppBot
 *
 *  Created by Alex on 1/31/10.
 *  Copyright 2010 __MyCompanyName__. All rights reserved.
 *
 */

#include <stdlib.h>
#include "CppBot_types.h"
#include "rob_pub.h"

#include "GLUT/glut.h"
#include "GLUTInterface.h"

#include "SimInterface.h"

#include "RobotBase.h"

unsigned int GLUTInterface::time_ms = 0;
unsigned int GLUTInterface::frame_period_ms = 5;
GLUTInterface::window_map_type GLUTInterface::window_map;     
bool GLUTInterface::running = true;

static bool s_joystick_works = false;
static uint32 s_joystick_buttons = 0;

static void keyboard_up_cb( unsigned char key, int x, int y );

/**************************************************************************
* Init()
*   Initialize GLUT functions
**************************************************************************/
void GLUTInterface::Init( unsigned int fps, int argc, char* argv[] )
{
/*----------------------------------------------------------
Compute frame period
----------------------------------------------------------*/
frame_period_ms = 1000 / fps;

/*----------------------------------------------------------
Initialize GLUT
----------------------------------------------------------*/
glutInit( &argc, argv );
glutInitDisplayMode( GLUT_RGB | GLUT_DOUBLE | GLUT_DEPTH );

/*----------------------------------------------------------
Register callbacks
----------------------------------------------------------*/
glutTimerFunc( frame_period_ms, time_proc, 0 );
}   /* GLUTInterface::Init */


/**************************************************************************
* RegisterWindow() - Call this function every time you create a window 
*   object
**************************************************************************/
void GLUTInterface::RegisterWindow( GLUTWindow * p_window )
{
window_map[ p_window->glut_id ] = p_window;

/*----------------------------------------------------------
Register callbacks
----------------------------------------------------------*/
glutReshapeFunc( reshape_cb );
glutDisplayFunc( display_cb );
glutKeyboardFunc( keyboard_cb );
glutKeyboardUpFunc( keyboard_up_cb );
//glutJoystickFunc( joystick_cb, frame_period_ms );
}   /* GLUTInterface::RegisterWindow */


/**************************************************************************
* display_cb( )
**************************************************************************/
void GLUTInterface::display_cb( void )
{
int cur_win_id = glutGetWindow( );
GLUTWindow * p_cur_win = window_map[ cur_win_id ];
_ASSERT( p_cur_win != NULL );
p_cur_win->DisplayCallBack( );
}   /* GLUTInterface::display_cb */


/**************************************************************************
* joystick_cb - Joystick callback function
**************************************************************************/
void GLUTInterface::joystick_cb
    ( 
    unsigned int        buttonMask, /* bitmask of pressed buttons           */
    int                 x,          /* x axis value [-1000,1000]            */
    int                 y,          /* y axis value [-1000,1000]            */
    int                 z           /* z axis value [-1000,1000]            */
    )
{
    for( int i = 0; i < SimInterface::JOYSTICK_CNT; i++ )
    {
        SimInterface::SetJoystick( i, 
                                   ( float )x / 1000.0f, 
                                   ( float )y / 1000.0f, 
                                   ( float )z / 1000.0f, 
                                   buttonMask << 1 );
    }
}   /* GLUTInterface::joystick_cb */


/**************************************************************************
* keyboard_cb( )
**************************************************************************/
void GLUTInterface::keyboard_cb
    (
    unsigned char       key,        /* key pressed                  */
    int                 x,          /* mouse X coordinate           */
    int                 y           /* mouse Y coordinate           */
    )
{
/*----------------------------------------------------------
Process keyboard key presses
----------------------------------------------------------*/
switch( key )
    {

    /*------------------------------------------------------
    Pause simulation
    ------------------------------------------------------*/
    case ' ':
        running = !running;
        break;

    /*------------------------------------------------------
    Start autonomous mode
    ------------------------------------------------------*/
    case 'a':
    case 'A':
        rob_set_mode( ROB_MODE_AUTO );
        time_ms = 0;
        break;
    
    /*------------------------------------------------------
    Start teleoperated mode
    ------------------------------------------------------*/
    case 't':
    case 'T':
        rob_set_mode( ROB_MODE_TELEOP );
        time_ms = 0;
        break;

    /*------------------------------------------------------
    Exit program if escape is pressed
    ------------------------------------------------------*/
    case 27:
        exit(0);
        break;

    /*------------------------------------------------------
    Default
    ------------------------------------------------------*/
    default:
        break;
    }
    
/*----------------------------------------------------------
If joystick does not work, use number keys for joystick 
buttons.
----------------------------------------------------------*/
if( !s_joystick_works )
    {
    if( ( key >= '0' ) && ( key <= '9' ) )
        {
        s_joystick_buttons = s_joystick_buttons | ( 1 << ( key - '0' ) );
        for( int i = 0; i < SimInterface::JOYSTICK_CNT; i++ )
            {
            SimInterface::SetJoystick( i, 0.0f, 0.0f, 0.0f, s_joystick_buttons );
            }
        }
    }
}   /* GLUTInterface::keyboard_cb */


/**************************************************************************
* reshape_cb - 
**************************************************************************/
void GLUTInterface::reshape_cb
    (
    int                 x,          /* new x dimension              */
    int                 y           /* new y demension              */
    )
{
int cur_win_id = glutGetWindow( );
GLUTWindow * p_cur_win = window_map[ cur_win_id ];
_ASSERT( p_cur_win != NULL );
p_cur_win->ReshapeCallBack( x, y );
}   /* GLUTInterface::reshape_cb */


/**************************************************************************
* time_proc - 
**************************************************************************/
void GLUTInterface::time_proc( int value )
{
// Restart the timer
glutTimerFunc( frame_period_ms, time_proc, 0 );

// Update the simulator
if( running )
    {
    time_ms += frame_period_ms;
    SimInterface::Update( frame_period_ms );
    if( RobotBase::p_robot != NULL )
        {
        RobotBase::p_robot->Update( frame_period_ms );
        }

    window_map_type::iterator i;
    for( i = window_map.begin(); i != window_map.end( ); i++ )
        {
        glutPostWindowRedisplay( i->first );
        }
    }
}   /* GLUTInterface::time_proc */


/**************************************************************************
* Keyboard up callback function
**************************************************************************/
static void keyboard_up_cb( unsigned char key, int x, int y )
{
/*----------------------------------------------------------
If joystick does not work, use number keys for joystick 
buttons.
----------------------------------------------------------*/
if( !s_joystick_works )
    {
    if( ( key >= '0' ) && ( key <= '9' ) )
        {
        s_joystick_buttons = s_joystick_buttons & ~( 1 << ( key - '0' ) );
        for( int i = 0; i < SimInterface::JOYSTICK_CNT; i++ )
            {
            SimInterface::SetJoystick( i, 0.0f, 0.0f, 0.0f, s_joystick_buttons );
            }
        }
    }
}
